/*
 * Decompiled with CFR 0.152.
 */
package com.skcraft.launcher.builder;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.ParameterException;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closer;
import com.google.common.io.Files;
import com.skcraft.launcher.Launcher;
import com.skcraft.launcher.LauncherUtils;
import com.skcraft.launcher.builder.BuilderConfig;
import com.skcraft.launcher.builder.BuilderOptions;
import com.skcraft.launcher.builder.BuilderUtils;
import com.skcraft.launcher.builder.ClientFileCollector;
import com.skcraft.launcher.builder.Compressor;
import com.skcraft.launcher.builder.FeaturePattern;
import com.skcraft.launcher.builder.FileInfoScanner;
import com.skcraft.launcher.builder.FileUrlScanner;
import com.skcraft.launcher.builder.JarFileFilter;
import com.skcraft.launcher.builder.PropertiesApplicator;
import com.skcraft.launcher.builder.loaders.FabricLoaderProcessor;
import com.skcraft.launcher.builder.loaders.ILoaderProcessor;
import com.skcraft.launcher.builder.loaders.LoaderResult;
import com.skcraft.launcher.builder.loaders.ModernForgeLoaderProcessor;
import com.skcraft.launcher.builder.loaders.OldForgeLoaderProcessor;
import com.skcraft.launcher.model.loader.BasicInstallProfile;
import com.skcraft.launcher.model.minecraft.Library;
import com.skcraft.launcher.model.minecraft.ReleaseList;
import com.skcraft.launcher.model.minecraft.Version;
import com.skcraft.launcher.model.minecraft.VersionManifest;
import com.skcraft.launcher.model.modpack.Manifest;
import com.skcraft.launcher.util.HttpRequest;
import com.skcraft.launcher.util.SimpleLogFormatter;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Properties;
import java.util.jar.JarFile;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import lombok.NonNull;

public class PackageBuilder {
    private static final Logger log = Logger.getLogger(PackageBuilder.class.getName());
    private final Properties properties;
    private final ObjectMapper mapper;
    private ObjectWriter writer;
    private final Manifest manifest;
    private final PropertiesApplicator applicator;
    private boolean prettyPrint = false;
    private File baseDir;
    private List<Library> loaderLibraries = Lists.newArrayList();
    private List<Library> installerLibraries = Lists.newArrayList();
    private List<String> mavenRepos;
    private List<URL> jarMavens = Lists.newArrayList();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PackageBuilder(@NonNull ObjectMapper mapper, @NonNull Manifest manifest) throws IOException {
        if (mapper == null) {
            throw new NullPointerException("mapper is marked non-null but is null");
        }
        if (manifest == null) {
            throw new NullPointerException("manifest is marked non-null but is null");
        }
        this.properties = LauncherUtils.loadProperties(Launcher.class, "launcher.properties", "com.skcraft.launcher.propertiesFile");
        this.mapper = mapper;
        this.manifest = manifest;
        this.applicator = new PropertiesApplicator(manifest);
        this.setPrettyPrint(false);
        try (Closer closer = Closer.create();){
            this.mavenRepos = mapper.readValue(closer.register(Launcher.class.getResourceAsStream("maven_repos.json")), new TypeReference<List<String>>(){});
        }
    }

    public void setPrettyPrint(boolean prettyPrint) {
        this.writer = prettyPrint ? this.mapper.writerWithDefaultPrettyPrinter() : this.mapper.writer();
        this.prettyPrint = prettyPrint;
    }

    public void scan(File dir) throws IOException {
        PackageBuilder.logSection("Scanning for .info.json files...");
        FileInfoScanner infoScanner = new FileInfoScanner(this.mapper);
        infoScanner.walk(dir);
        for (FeaturePattern pattern : infoScanner.getPatterns()) {
            this.applicator.register(pattern);
        }
        PackageBuilder.logSection("Scanning for .url.txt files...");
        FileUrlScanner urlScanner = new FileUrlScanner();
        urlScanner.walk(dir);
    }

    public void addFiles(File dir, File destDir) throws IOException {
        PackageBuilder.logSection("Adding files to modpack...");
        ClientFileCollector collector = new ClientFileCollector(this.manifest, this.applicator, destDir);
        collector.walk(dir);
    }

    public void addLoaders(File dir, File librariesDir) {
        PackageBuilder.logSection("Checking for mod loaders to install...");
        LinkedHashSet<Library> collected = new LinkedHashSet<Library>();
        File[] files = dir.listFiles(new JarFileFilter());
        if (files != null) {
            for (File file : files) {
                try {
                    this.processLoader(collected, file, librariesDir);
                }
                catch (IOException e) {
                    log.log(Level.WARNING, "Failed to add the loader at " + file.getAbsolutePath(), e);
                }
            }
        }
        this.loaderLibraries.addAll(collected);
        VersionManifest version = this.manifest.getVersionManifest();
        collected.addAll(version.getLibraries());
        version.setLibraries(collected);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processLoader(LinkedHashSet<Library> loaderLibraries, File file, File librariesDir) throws IOException {
        log.info("Installing " + file.getName() + "...");
        JarFile jarFile = new JarFile(file);
        Closer closer = Closer.create();
        ILoaderProcessor processor = null;
        try {
            ZipEntry profileEntry = BuilderUtils.getZipEntry(jarFile, "install_profile.json");
            if (profileEntry != null) {
                InputStream stream = jarFile.getInputStream(profileEntry);
                InputStreamReader reader = closer.register(new InputStreamReader(stream));
                BasicInstallProfile basicProfile = this.mapper.readValue(BuilderUtils.readStringFromStream(reader), BasicInstallProfile.class);
                processor = basicProfile.isLegacy() ? new OldForgeLoaderProcessor() : new ModernForgeLoaderProcessor();
            } else if (BuilderUtils.getZipEntry(jarFile, "fabric-installer.json") != null) {
                processor = new FabricLoaderProcessor(FabricLoaderProcessor.Variant.FABRIC);
            } else if (BuilderUtils.getZipEntry(jarFile, "quilt_installer.json") != null) {
                processor = new FabricLoaderProcessor(FabricLoaderProcessor.Variant.QUILT);
            }
        }
        finally {
            closer.close();
            jarFile.close();
        }
        if (processor != null) {
            LoaderResult result = processor.process(file, this.manifest, this.mapper, this.baseDir);
            if (result == null) {
                log.warning("Loader " + file.getName() + " failed to process.");
                return;
            }
            loaderLibraries.addAll(result.getLoaderLibraries());
            this.installerLibraries.addAll(result.getProcessorLibraries());
            this.jarMavens.addAll(result.getJarMavens());
        } else {
            log.warning("Loader " + file.getName() + " was skipped due to missing metadata. Is it really a loader JAR?");
        }
    }

    public void downloadLibraries(File librariesDir) throws IOException, InterruptedException {
        PackageBuilder.logSection("Downloading libraries...");
        for (Library library : Iterables.concat(this.loaderLibraries, this.installerLibraries)) {
            library.ensureDownloadsExist();
            for (Library.Artifact artifact : library.getDownloads().getAllArtifacts()) {
                Iterator<Object> iterator;
                File outputPath = new File(librariesDir, artifact.getPath());
                if (outputPath.exists()) continue;
                Files.createParentDirs(outputPath);
                boolean found = false;
                boolean urlEmpty = artifact.getUrl().isEmpty();
                if (!urlEmpty && !artifact.getUrl().endsWith("/")) {
                    found = this.tryDownloadLibrary(library, artifact, artifact.getUrl(), outputPath);
                }
                if (!found) {
                    URL base;
                    iterator = this.jarMavens.iterator();
                    while (iterator.hasNext() && !(found = this.tryFetchLibrary(library, new URL(base = iterator.next(), artifact.getPath()), outputPath))) {
                    }
                }
                if (!found && !urlEmpty) {
                    URL url = LauncherUtils.concat(HttpRequest.url(artifact.getUrl()), artifact.getPath());
                    found = this.tryDownloadLibrary(library, artifact, url.toString(), outputPath);
                }
                if (!found) {
                    String baseUrl;
                    iterator = this.mavenRepos.iterator();
                    while (iterator.hasNext() && !(found = this.tryDownloadLibrary(library, artifact, (baseUrl = (String)iterator.next()) + artifact.getPath(), outputPath))) {
                    }
                }
                if (found) continue;
                log.warning("!! Failed to download the library " + library.getName() + " -- this means your copy of the libraries will lack this file");
            }
        }
    }

    private boolean tryDownloadLibrary(Library library, Library.Artifact artifact, String baseUrl, File outputPath) throws IOException, InterruptedException {
        URL url = new URL(baseUrl);
        if (url.getPath().isEmpty() || url.getPath().equals("/")) {
            return false;
        }
        List<Compressor> compressors = BuilderUtils.getCompressors(baseUrl);
        for (Compressor compressor : Lists.reverse(compressors)) {
            url = new URL(compressor.transformPathname(url.toString()));
        }
        File tempFile = File.createTempFile("launcherlib", null);
        try {
            log.info("Downloading library " + library.getName() + " from " + url + "...");
            HttpRequest.get(url).execute().expectResponseCode(200).expectContentType("application/java-archive", "application/octet-stream", "application/zip").saveContent(tempFile);
        }
        catch (IOException e) {
            log.info("Could not get file from " + url + ": " + e.getMessage());
            return false;
        }
        this.writeLibraryToFile(outputPath, tempFile, compressors);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean tryFetchLibrary(Library library, URL url, File outputPath) throws IOException {
        File tempFile = File.createTempFile("launcherlib", null);
        try (Closer closer = Closer.create();){
            log.info("Reading library " + library.getName() + " from " + url.toString());
            InputStream stream = closer.register(url.openStream());
            stream = closer.register(new BufferedInputStream(stream));
            ByteStreams.copy(stream, (OutputStream)closer.register(new FileOutputStream(tempFile)));
        }
        this.writeLibraryToFile(outputPath, tempFile, Collections.emptyList());
        return true;
    }

    private void writeLibraryToFile(File outputPath, File inputFile, List<Compressor> compressors) throws IOException {
        Closer closer = Closer.create();
        InputStream inputStream = closer.register(new FileInputStream(inputFile));
        inputStream = closer.register(new BufferedInputStream(inputStream));
        for (Compressor compressor : compressors) {
            inputStream = closer.register(compressor.createInputStream(inputStream));
        }
        ByteStreams.copy(inputStream, (OutputStream)closer.register(new FileOutputStream(outputPath)));
        inputFile.delete();
        closer.close();
    }

    public void validateManifest() {
        Preconditions.checkNotNull(Strings.emptyToNull(this.manifest.getName()), "Package name is not defined");
        Preconditions.checkNotNull(Strings.emptyToNull(this.manifest.getGameVersion()), "Game version is not defined");
    }

    public void readConfig(File path) throws IOException {
        if (path != null) {
            BuilderConfig config = this.read(path, BuilderConfig.class);
            config.update(this.manifest);
            config.registerProperties(this.applicator);
        }
    }

    public void readVersionManifest(File path) throws IOException, InterruptedException {
        PackageBuilder.logSection("Reading version manifest...");
        if (path.exists()) {
            VersionManifest versionManifest = this.read(path, VersionManifest.class);
            this.manifest.setVersionManifest(versionManifest);
            log.info("Loaded version manifest from " + path.getAbsolutePath());
        } else {
            URL url = HttpRequest.url(this.properties.getProperty("versionManifestUrl"));
            log.info("Fetching version manifest from " + url + "...");
            ReleaseList releases = HttpRequest.get(url).execute().expectResponseCode(200).returnContent().asJson(ReleaseList.class);
            Version version = releases.find(this.manifest.getGameVersion());
            VersionManifest versionManifest = HttpRequest.get(HttpRequest.url(version.getUrl())).execute().expectResponseCode(200).returnContent().asJson(VersionManifest.class);
            this.manifest.setVersionManifest(versionManifest);
        }
    }

    public void writeManifest(@NonNull File path) throws IOException {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        PackageBuilder.logSection("Writing manifest...");
        this.manifest.setFeatures(this.applicator.getFeaturesInUse());
        VersionManifest versionManifest = this.manifest.getVersionManifest();
        if (versionManifest != null) {
            versionManifest.setId(this.manifest.getGameVersion());
        }
        this.validateManifest();
        path.getAbsoluteFile().getParentFile().mkdirs();
        this.writer.writeValue(path, (Object)this.manifest);
        log.info("Wrote manifest to " + path.getAbsolutePath());
    }

    private static BuilderOptions parseArgs(String[] args) {
        BuilderOptions options = new BuilderOptions();
        new JCommander(options).parse(args);
        options.choosePaths();
        return options;
    }

    private <V> V read(File path, Class<V> clazz) throws IOException {
        try {
            if (path == null) {
                return clazz.newInstance();
            }
            return this.mapper.readValue(path, clazz);
        }
        catch (InstantiationException e) {
            throw new IOException("Failed to create " + clazz.getCanonicalName(), e);
        }
        catch (IllegalAccessException e) {
            throw new IOException("Failed to create " + clazz.getCanonicalName(), e);
        }
    }

    public static void main(String[] args) throws IOException, InterruptedException {
        BuilderOptions options;
        try {
            options = PackageBuilder.parseArgs(args);
        }
        catch (ParameterException e) {
            new JCommander().usage();
            System.err.println("error: " + e.getMessage());
            System.exit(1);
            return;
        }
        SimpleLogFormatter.configureGlobalLogger();
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_ABSENT);
        Manifest manifest = new Manifest();
        manifest.setMinimumVersion(3);
        PackageBuilder builder = new PackageBuilder(mapper, manifest);
        builder.setPrettyPrint(options.isPrettyPrinting());
        builder.readConfig(options.getConfigPath());
        builder.readVersionManifest(options.getVersionManifestPath());
        builder.setBaseDir(options.getOutputPath());
        manifest.updateName(options.getName());
        manifest.updateTitle(options.getTitle());
        manifest.updateGameVersion(options.getGameVersion());
        manifest.setVersion(options.getVersion());
        manifest.setLibrariesLocation(options.getLibrariesLocation());
        manifest.setObjectsLocation(options.getObjectsLocation());
        builder.scan(options.getFilesDir());
        builder.addFiles(options.getFilesDir(), options.getObjectsDir());
        builder.addLoaders(options.getLoadersDir(), options.getLibrariesDir());
        builder.downloadLibraries(options.getLibrariesDir());
        builder.writeManifest(options.getManifestPath());
        PackageBuilder.logSection("Done");
        log.info("Now upload the contents of " + options.getOutputPath() + " to your web server or CDN!");
    }

    private static void logSection(String name) {
        log.info("");
        log.info("--- " + name + " ---");
    }

    public boolean isPrettyPrint() {
        return this.prettyPrint;
    }

    public File getBaseDir() {
        return this.baseDir;
    }

    public void setBaseDir(File baseDir) {
        this.baseDir = baseDir;
    }
}

